<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Hashtag\Post;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Models\Hashtag\HashtagModel;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagPostTable;

final class HashtagPostContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function load(string $feedType, string $loadType): void
    {
        self::feedHelperInit($feedType, $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingForHashtagPage())
            {
                $hashtagIdFromReq = self::$request -> findKey(
                    HashtagPostTable::HASHTAG_ID,
                    RequestTable::PAYLOAD,
                    HashtagPostTable::TABLE_NAME
                );

                self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $hashtagIdFromReq);
            }

            $offset = self::$request -> findKeyOffset(HashtagPostTable::POST_ID, HashtagPostTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            |  ensure target model exists, if hashtag page
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingForHashtagPage())
            {
                /**
                 * flag: not used anywhere else
                 */
                $targetHashtagModel = HashtagModel::findFromId_noException($hashtagIdFromReq);

                self::ensureModel($targetHashtagModel);
            }

            /*
            |--------------------------------------------------------------------------
            | for some magic, put it in a while
            |--------------------------------------------------------------------------
            */

            while (true)
            {
                /*
                |--------------------------------------------------------------------------
                | query builder
                |--------------------------------------------------------------------------
                */

                $hashtagPostTableQuery = (new Query()) -> from(HashtagPostTable::TABLE_NAME);

                /*
                |--------------------------------------------------------------------------
                | selection order
                |--------------------------------------------------------------------------
                */

                if (self::isLoadingLatestContent())
                {
                    $hashtagPostTableQuery -> greaterThan(HashtagPostTable::POST_ID, $offset);
                }
                else
                {
                    $hashtagPostTableQuery -> lessThan(HashtagPostTable::POST_ID, $offset);
                }

                /*
                |--------------------------------------------------------------------------
                | loading for hashtag page
                |--------------------------------------------------------------------------
                */

                if (self::isLoadingForHashtagPage())
                {
                    /*
                    |--------------------------------------------------------------------------
                    | fetch only for single hashtag(id)
                    |--------------------------------------------------------------------------
                    */

                    $hashtagPostTableQuery -> where(HashtagPostTable::HASHTAG_ID, $hashtagIdFromReq);
                }

                /*
                |--------------------------------------------------------------------------
                | for explore page
                |--------------------------------------------------------------------------
                */
                elseif (self::isLoadingForExplorePage())
                {
                    /*
                    |--------------------------------------------------------------------------
                    | fetch user's hashtag following list
                    |--------------------------------------------------------------------------
                    */

                    $followingHashtagIds = CommonQueries::userHashtagFollowingIds(self::$authedUserModel -> getId());

                    /*
                    |--------------------------------------------------------------------------
                    | fetch for hashtags that user is following
                    |--------------------------------------------------------------------------
                    */

                    $hashtagPostTableQuery -> in(HashtagPostTable::HASHTAG_ID, $followingHashtagIds);
                }

                /*
                |--------------------------------------------------------------------------
                | order by post id & limit
                |--------------------------------------------------------------------------
                */

                $hashtagPostTableQuery
                    -> orderByDesc(HashtagPostTable::POST_ID)
                    -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_HASHTAG_POST));

                /*
                |--------------------------------------------------------------------------
                | select distinct only post ids(and user ids)
                |--------------------------------------------------------------------------
                */

                $hashtagPostTableQuery
                    -> whereDistinct(HashtagPostTable::POST_ID)
                    -> whereDistinct(HashtagPostTable::POST_OWNER_USER_ID);

                /*
                |--------------------------------------------------------------------------
                | fetch
                |--------------------------------------------------------------------------
                */

                $dataRows = $hashtagPostTableQuery -> selectDistinctOnlyFields();

                /*
                |--------------------------------------------------------------------------
                | check end of results
                |--------------------------------------------------------------------------
                */

                if (0 == \count($dataRows))
                {
                    return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
                }

                /*
                |--------------------------------------------------------------------------
                | prepare fetched user ids and post ids
                |--------------------------------------------------------------------------
                */

                $userIdToPostIdsMap = array();

                foreach ($dataRows as $dataRow)
                {
                    $userId = $dataRow[HashtagPostTable::POST_OWNER_USER_ID];
                    $postId = $dataRow[HashtagPostTable::POST_ID];

                    /*
                    |--------------------------------------------------------------------------
                    | update offset
                    |--------------------------------------------------------------------------
                    */

                    if (self::isLoadingLatestContent())
                    {
                        if ($offset < $postId)
                        {
                            $offset = $postId;
                        }
                    }
                    else
                    {
                        if ($offset > $postId)
                        {
                            $offset = $postId;
                        }
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | update post ids map
                    |--------------------------------------------------------------------------
                    */

                    if ( ! isset($userIdToPostIdsMap[$userId]))
                    {
                        $userIdToPostIdsMap[$userId] = array($postId);

                        continue;
                    }

                    $userIdToPostIdsMap[$userId][] = $postId;
                }

                /*
                |--------------------------------------------------------------------------
                | fetch users first, do pirvacy checks here
                |--------------------------------------------------------------------------
                */

                /**
                 * @var UserModel[]
                 */
                $fetchedUserModels = array();

                /**
                 * @var UserModel[]
                 */
                $userModelsFromQuery = CommonQueries::modelsWithMatchingPredicates(
                    UserTable::getTableName(),
                    array(
                        UserTable::ID => \array_keys($userIdToPostIdsMap),
                    )
                );

                foreach ($userModelsFromQuery as $userModel)
                {
                    if (UserModelHelper::isUserContentAvailable($userModel, self::$authedUserModel))
                    {
                        $fetchedUserModels[] = $userModel;
                    }
                    else
                    {
                        // remove user content from results

                        unset($userIdToPostIdsMap[$userModel -> getId()]);
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | whether after privacy checks we've something to respond or not?
                |--------------------------------------------------------------------------
                */

                if (0 == \count($fetchedUserModels))
                {
                    // do nothing, let the while loop run and try adding something to content
                }
                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | add users to response
                    |--------------------------------------------------------------------------
                    */

                    foreach ($fetchedUserModels as $userModel)
                    {
                        self::addToResponse(UserTable::getTableName(), $userModel -> getDataMap());
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | fetch and add posts to response
                    |--------------------------------------------------------------------------
                    */

                    self::fetchModelsAndAdd(
                        PostTable::getTableName(),
                        array(
                            PostTable::ID => \array_merge(...\array_values($userIdToPostIdsMap)),
                        )
                    );

                    /*
                    |--------------------------------------------------------------------------
                    | additional data | user follow maps
                    | -------------------------------------------------------------------------
                    | help build follow button
                    |--------------------------------------------------------------------------
                    */

                    $usersContainer = self::$dataDock -> getContainer(UserTable::getTableName());

                    self::fetchModelsAndAddAsAdditional(
                        UserFollowTable::getTableName(),
                        array(
                            UserFollowTable::FOLLOWED_USER_ID    => $usersContainer -> getIds(),
                            UserFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                        )
                    );

                    /*
                    |--------------------------------------------------------------------------
                    | additional data | post likes maps
                    | -------------------------------------------------------------------------
                    | help build like button
                    |--------------------------------------------------------------------------
                    */

                    $postsContainer = self::$dataDock -> getContainer(PostTable::getTableName());

                    self::fetchModelsAndAddAsAdditional(
                        PostLikeTable::getTableName(),
                        array(
                            PostLikeTable::LIKED_POST_ID    => $postsContainer -> getIds(),
                            PostLikeTable::LIKED_BY_USER_ID => self::$authedUserModel -> getId(),
                        )
                    );

                    /*
                    |--------------------------------------------------------------------------
                    | additional data | post save maps
                    | -------------------------------------------------------------------------
                    | help build post save button
                    |--------------------------------------------------------------------------
                    */

                    self::fetchModelsAndAddAsAdditional(
                        PostSaveTable::getTableName(),
                        array(
                            PostSaveTable::SAVED_POST_ID    => $postsContainer -> getIds(),
                            PostSaveTable::SAVED_BY_USER_ID => self::$authedUserModel -> getId(),
                        )
                    );

                    /*
                    |--------------------------------------------------------------------------
                    | break the loop
                    |--------------------------------------------------------------------------
                    */

                    break;
                }
            }
        });
    }
}
